# -*- coding: utf-8 -*-
##########################################################################
#
#   Copyright (c) 2015-Present Webkul Software Pvt. Ltd. (<https://webkul.com/>)
#   See LICENSE file for full copyright and licensing details.
#   License URL : <https://store.webkul.com/license.html/>
#
##########################################################################

from werkzeug.urls import url_join
from odoo import api, fields, _
from odoo.tools.misc import get_lang
import logging
from odoo.addons.seo_url_redirect.models.model import Models
from odoo.addons.seo_url_redirect.models.ir_http import slug as http_slug
from odoo.http import request
_logger = logging.getLogger(__name__)


class ProductProduct(Models):

    _inherit = "product.product"

    def update_seo_url(self):
        for rec in self:
            rec.product_tmpl_id.update_seo_url()

    def open_redirect_urls(self):
        for rec in self:
            action = self.env['ir.actions.act_window']._for_xml_id(
                'website.action_website_rewrite_list')
            action['domain'] = [('record_id', '=', rec.product_tmpl_id.id),
                                ('rewrite_val', '=', 'product.template'),]
            return action

    @api.depends_context('lang')
    @api.depends('product_tmpl_id.website_url', 'product_template_attribute_value_ids')
    def _compute_product_website_url(self):
        seo_config = self.env['seo.rewrite.config'].sudo().search(
            [('state', '=', 'enabled'), ('seo_model', '=', 'product')], limit=1)
        if seo_config.remove_attr_params:
            for product in self:
                url = product.product_tmpl_id.website_url
                product.website_url = url
        else:
            super(ProductProduct, self)._compute_product_website_url()


class ProductTemplate(Models):

    _inherit = "product.template"

    url_redirect_generated = fields.Boolean(
        "redirect generated", default=False)

    url_key = fields.Char(string='SEO Url Key', default='',
                          translate=True, help="SEO Url Key for Product", copy=False)

    def update_field_translations(self, field_name, translations):
        """ Update the values of a translated field.

        :param str field_name: field name
        :param dict translations: if the field has ``translate=True``, it should be a dictionary
            like ``{lang: new_value}``; if ``translate`` is a callable, it should be like
            ``{lang: {old_term: new_term}}``
        """
        seo_config = self.env['seo.rewrite.config'].sudo().search(
            [('state', '=', 'enabled'), ('seo_model', '=', 'product')], limit=1)
        if seo_config and field_name == 'url_key':
            for key, value in translations.items():
                self.env['website.rewrite'].sudo(
                ).updateSeoUrlKey(self, key, {field_name: value})
        return super(ProductTemplate, self).update_field_translations(field_name, translations)

    def ir_translation_server_action(self):
        default_lang = get_lang(self.env).mapped('code')
        installed_languages = [lang[0]
                               for lang in self.env['res.lang'].get_installed()]

        # *********************************************************
        # *       Update terms for products                       *
        # *********************************************************

        all_product_tmpl = self
        for product in all_product_tmpl:
            translation = self.env['ir.translation'].search(
                [('res_id', '=', product.id), ('src', '=', product.url_key)])
            without_value_translation = translation.filtered(
                lambda r: r if not r.value else None)
            all_lang = installed_languages.copy()
            languages_without_terms = [lang for lang in all_lang if (
                lang not in translation.mapped('lang'))]
            default_lang_data = translation.filtered(
                lambda r: r.lang == default_lang[0])
            if default_lang_data:
                default_lang_data = default_lang_data[0]
            if languages_without_terms and default_lang_data and default_lang_data.src:
                for lang in languages_without_terms:
                    self.env['ir.translation'].create({
                        'type': default_lang_data.type,
                        'name': default_lang_data.name,
                        'module': default_lang_data.module,
                        'lang': lang,
                        'res_id': default_lang_data.res_id,
                        'value': default_lang_data.value,
                        'state': default_lang_data.state,
                        'src': default_lang_data.src,
                    })
            if without_value_translation:
                for rec in without_value_translation:
                    if default_lang_data.value:
                        rec.write({'value': default_lang_data.value,
                                  'state': default_lang_data.state, })

    @api.model_create_multi
    def create(self, vals):
        res = super(ProductTemplate, self).create(vals)
        seo_config = self.env['seo.rewrite.config'].sudo().search(
            [('seo_model', '=', 'product'), ('state', '=', 'enabled')], limit=1)
        if seo_config and res.url_key in ['', False, None]:
            self.env['website.rewrite'].sudo().setSeoUrlKey(
                'pattern_product', res)
        return res

    def write(self, vals):
        res = super(ProductTemplate, self).write(vals)
        seo_config = self.env['seo.rewrite.config'].sudo().search(
            [('seo_model', '=', 'product')], limit=1)
        if seo_config and vals.get('url_key') and self._context.get('lang'):
            if self._context.get('current_lang'):
                lang = self._context.get('current_lang')
            else:
                lang = self._context['lang']
            self.env['website.rewrite'].sudo(
            ).updateSeoUrlKey(self, lang, vals)

        return res

    def update_seo_url(self):
        seo_config = self.env['seo.rewrite.config'].sudo().search(
            [('seo_model', '=', 'product')], limit=1)
        if seo_config:
            return self.keep_original_keys()
        else:
            return self.env['wk.wizard.message'].genrated_message(
                _('Please enable the product seo configuration first.'))

    def keep_original_keys(self):
        active_objs = False
        vals = {'pattern': 'pattern_product'}
        active_ids = self._ids if self else self._context.get('active_ids')
        active_objs = self.env['product.template'].search(
            [('id', 'in', active_ids)])
        installed_langs = self.env['website.rewrite'].sudo(
        ).get_installed_langs()
        for lang in installed_langs:
            self.env['website.rewrite'].sudo().updateSeoUrlKey(
                active_objs, lang=lang.code, vals=vals)
        text = "SEO Url key of {} records have been successfully updated.".format(
            len(active_ids))
        return self.env['wk.wizard.message'].genrated_message(text)

    def open_website_url(self):
        '''Execute when we click on open website from product template smart button'''
        seo_config = self.env['seo.rewrite.config'].sudo().search(
            [('state', '=', 'enabled'), ('seo_model', '=', 'product')], limit=1)
        if request.httprequest.cookies.get('frontend_lang'):
            self = self.with_context(
                lang=request.httprequest.cookies.get('frontend_lang'))
        res = super(ProductTemplate, self).open_website_url()
        return res

    def _compute_website_url(self):
        '''Execute when we open /shop,any product from backend and frontented.'''
        seo_config = self.env['seo.rewrite.config'].sudo().search(
            [('state', '=', 'enabled'), ('seo_model', '=', 'product')], limit=1)
        super(ProductTemplate, self)._compute_website_url()
        if seo_config:

            for product in self:
                if product.id:
                    seo_config.get_seo_product_url(product)
        else:
            super()._compute_website_url()

    def open_redirect_urls(self):
        action = self.env['ir.actions.act_window']._for_xml_id(
            'website.action_website_rewrite_list')
        action['domain'] = [('record_id', '=', self.id),
                            ('rewrite_val', '=', 'product.template'),]
        return action

    def _get_product_display_url(self, seo_config):
        hierarchy = self._context.get('hierarchy')
        path = request.httprequest.path
        website_id = self.env["website"].get_current_website()
        base_url = request.httprequest.url_root
        lang = self._context.get('lang')
        lang_obj = self.env['res.lang']
        if lang not in [website_id.default_lang_id.code, website_id.default_lang_id.url_code]:
            lang_obj = self.env['res.lang'].search(
                ['|', ('code', '=', lang), ('url_code', '=', lang)], limit=1)
        if hierarchy:
            url = hierarchy.get(path, '/')
            new_lang = ''
            prefix = ''
            if not seo_config.use_server_rewrites:
                prefix = 'shop'
            if lang_obj:
                new_lang = lang_obj.url_code+'/'
            if new_lang and new_lang in url:
                if prefix:
                    prefix = prefix+'/'
                url = url.replace(new_lang, new_lang+prefix)
            elif prefix:
                url = '/'+prefix+url
            return base_url, url
        return base_url, self.website_url

    def _default_website_meta(self):
        res = super(ProductTemplate, self)._default_website_meta()
        seo_config = self.env['seo.rewrite.config'].sudo().search(
            [('state', '=', 'enabled'), ('seo_model', '=', 'product')], limit=1)
        if seo_config:
            for product in self:
                base_url, url = product._get_product_display_url(seo_config)
                res['default_opengraph']['og:url'] = url_join(base_url, url)
        return res

    def _get_seo_switcher_url(self, seo_config):
        return self.website_url
