# -*- coding: utf-8 -*-
##########################################################################
#
#   Copyright (c) 2015-Present Webkul Software Pvt. Ltd. (<https://webkul.com/>)
#   See LICENSE file for full copyright and licensing details.
#   License URL : <https://store.webkul.com/license.html/>
#
##########################################################################

from werkzeug.urls import url_join
from odoo.tools.misc import formatLang, format_date as odoo_format_date, get_lang
from odoo import api, fields, models, _
from odoo.http import request

from odoo.addons.seo_url_redirect.models.model import Models
import logging
_logger = logging.getLogger(__name__)


class ProductPublicCategory(Models):

    _inherit = "product.public.category"

    url_key = fields.Char(string='SEO Url Key', default='', translate=True,
                          help="SEO Url Key for Product Category", copy=False)

    def update_field_translations(self, field_name, translations):
        """ Update the values of a translated field.

        :param str field_name: field name
        :param dict translations: if the field has ``translate=True``, it should be a dictionary
            like ``{lang: new_value}``; if ``translate`` is a callable, it should be like
            ``{lang: {old_term: new_term}}``
        """
        if field_name == 'url_key':
            for key, value in translations.items():
                if value:
                    self.env['website.rewrite'].sudo(
                    ).updateSeoUrlKey(self, key, {field_name: value})
        return super(ProductPublicCategory, self).update_field_translations(field_name, translations)


    @api.model_create_multi
    def create(self, vals):
        res = super(ProductPublicCategory, self).create(vals)
        seo_config = self.env['seo.rewrite.config'].sudo().search(
            [('seo_model', '=', 'category')], limit=1)
        if seo_config and res.url_key in ['', False, None]:
            self.env['website.rewrite'].setSeoUrlKey('pattern_category', res)
        return res

    def write(self, vals):
        res = super(ProductPublicCategory, self).write(vals)
        seo_config = self.env['seo.rewrite.config'].sudo().search(
            [('seo_model', '=', 'category')], limit=1)
        if seo_config and vals.get('url_key') and self._context.get('lang'):
            if self._context.get('current_lang'):
                lang = self._context['current_lang']
            else:
                lang = self._context['lang']
            self.env['website.rewrite'].sudo(
            ).updateSeoUrlKey(self, lang, vals)
        return res

    def update_seo_url(self):

        seo_config = self.env['seo.rewrite.config'].sudo().search(
            [('seo_model', '=', 'category')], limit=1)
        if seo_config:
            return self.keep_original_keys()
        else:
            return self.env['wk.wizard.message'].genrated_message(_('Please enable the category seo configuration first.'))

    def keep_original_keys(self):
        active_objs = False
        vals = {'pattern': 'pattern_category'}
        active_ids = self._ids if self else self._context.get('active_ids')
        active_objs = self.env['product.public.category'].search(
            [('id', 'in', active_ids)])
        installed_langs = self.env['website.rewrite'].sudo(
        ).get_installed_langs()
        for lang in installed_langs:
            self.env['website.rewrite'].sudo().updateSeoUrlKey(
                active_objs, lang.code, vals=vals)
        text = "SEO Url key of {} records have been successfully updated.".format(
            len(active_ids))
        return self.env['wk.wizard.message'].genrated_message(text)

    def open_redirect_urls(self):
        action = self.env['ir.actions.act_window']._for_xml_id(
            'website.action_website_rewrite_list')
        action['domain'] = [('record_id', '=', self.id),
                            ('rewrite_val', '=', 'product.public.category'),]
        return action

    def _get_category_display_url(self, seo_config):
        hierarchy = self._context.get('hierarchy')
        path = request.httprequest.path
        website_id = self.env["website"].get_current_website()
        base_url = request.httprequest.url_root
        lang = self._context.get('lang')
        lang_obj = self.env['res.lang']
        if lang not in [website_id.default_lang_id.code, website_id.default_lang_id.url_code]:
            lang_obj = self.env['res.lang'].search(
                ['|', ('code', '=', lang), ('url_code', '=', lang)], limit=1)
        if hierarchy:
            url = hierarchy.get(path, '/')
            new_lang = ''
            prefix = ''
            if not seo_config.use_server_rewrites:
                prefix = 'shop/category'
            if lang_obj:
                new_lang = lang_obj.url_code+'/'
            if new_lang and new_lang in url:
                if prefix:
                    prefix = prefix+'/'
                url = url.replace(new_lang, new_lang+prefix)
            elif prefix:
                url = '/'+prefix+url
            return base_url, url

    def _default_website_meta(self):
        res = super(ProductPublicCategory, self)._default_website_meta()
        seo_config = self.env['seo.rewrite.config'].sudo().search(
            [('state', '=', 'enabled'), ('seo_model', '=', 'category')], limit=1)
        if seo_config:
            for cat in self:
                base_url, path = self._get_category_display_url(seo_config)
                res['default_opengraph']['og:url'] = url_join(base_url, path)
        return res

    def _get_seo_switcher_url(self, seo_config):

        return seo_config.getCategoryUrl(self)
