import logging

from odoo import models, api, fields

_logger = logging.getLogger(__name__)

import threading

# Thread-local flag to detect internal update
_thread_local = threading.local()


class ProductProduct(models.Model):
    _inherit = 'product.product'

    # Redefine lst_price to remove the compute method
    lst_price = fields.Float(
        string='Sales Price',
        digits='Product Price',
        help="The sale price is managed from the product template.",
        compute=False,
        store=True  # Use True if you want this to be saved to DB
    )

    def write(self, vals):
        if 'list_price' in vals and getattr(_thread_local, 'bypass_variant_sync', False):
            _logger.info("🛑 Blocking list_price write on variant [%s] due to bypass flag", self.display_name)
            vals.pop('list_price', None)

        res = super().write(vals)

        for rec in self:
            if 'price_extra' in vals:
                rec.lst_price = rec.price_extra
                _logger.info("✅ Setting lst_price = price_extra (%.2f) on variant [%s]", rec.price_extra,
                             rec.display_name)

        return res

    def update_all_lst_prices(self):
        for variant in self:
            template = variant.product_tmpl_id
            if template and len(template.product_variant_ids) > 1:
                variant.lst_price = variant.price_extra


class ProductTemplateAttributeValue(models.Model):
    _inherit = 'product.template.attribute.value'

    def write(self, vals):
        res = super().write(vals)
        if 'price_extra' in vals:
            for variant in self.product_tmpl_id.product_variant_ids:
                variant.lst_price = variant.price_extra

            ptavs = self.env['product.template.attribute.value'].search([
                ('product_tmpl_id', '=', self.product_tmpl_id.id)
            ])
            valid_ptavs = ptavs.filtered(lambda p: p.price_extra > 0)
            if valid_ptavs:
                variant_min_price = min(valid_ptavs.mapped('price_extra'))
                self.product_tmpl_id.list_price = variant_min_price
        return res

    def _update_template_price(self):
        for ptav in self:
            template = ptav.product_tmpl_id
            if not template:
                continue
            all_ptavs = self.search([('product_tmpl_id', '=', template.id)])
            non_zero_prices = [p for p in all_ptavs.mapped('price_extra') if p > 0]

            if non_zero_prices:
                new_price = min(non_zero_prices)
                _logger.info("💡 _update_template_price: Setting list_price for [%s] to %.2f", template.name,
                             new_price)
                template.list_price = new_price
            else:
                _logger.info("💡 _update_template_price: No valid prices, setting list_price to 0 for [%s]",
                             template.name)
                template.list_price = 0.0


class ProductTemplate(models.Model):
    _inherit = 'product.template'

    @api.model
    def update_template_price_daily(self):
        all_templates = self.search([('product_variant_ids', '!=', False)])

        _thread_local.bypass_variant_sync = True

        for template in all_templates:
            # Get all attribute values linked to this template
            ptavs = self.env['product.template.attribute.value'].search([
                ('product_tmpl_id', '=', template.id)
            ])

            # Filter out zero or negative extra prices
            valid_price_extras = [p for p in ptavs.mapped('price_extra') if p > 0]

            # Set the template's sale price to the minimum non-zero price_extra
            if valid_price_extras:
                template.list_price = min(valid_price_extras)
            else:
                # Optional: Skip or log if no positive price found
                # _logger.info(f"No valid price_extra > 0 found for template {template.name}")
                pass

        _thread_local.bypass_variant_sync = False

    def _update_template_price_from_attrs(self):
        for template in self:
            ptavs = self.env['product.template.attribute.value'].search([
                ('product_tmpl_id', '=', template.id)
            ])
            valid_ptavs = ptavs.filtered(lambda p: p.price_extra > 0)

            _logger.info("Template: %s, Valid Extras: %s", template.name, valid_ptavs.mapped('price_extra'))

            if valid_ptavs:
                new_price = min(valid_ptavs.mapped('price_extra'))
                # template.write({'list_price': new_price})
                template.list_price = new_price
                _logger.info("✅ Template [%s] list_price set to %.2f", template.name, new_price)
            # else:
            #     template.write({'list_price': 0.0})

            # Always sync lst_price
            for variant in template.product_variant_ids:
                variant.lst_price = variant.price_extra
                _logger.info("🔁 Updated variant [%s] lst_price = %.2f", variant.name, variant.lst_price)
